/**
 * A custom script for the Platypush browser extension that simplifies the
 * current page by extracting the main content and displaying it in a clean
 * format.
 *
 * This script uses the Mercury parser to extract the main content of the page
 * and formats it for better readability.
 *
 * The parsed content will be added inside a new <div> element with a
 * simplified style, and the previous page content will be hidden.
 *
 * (c) 2025 Fabio Manganiello <fabio@manganiello.tech>
 * @license GNU General Public License v3.0
 */

// Entry point for the script. All the logic should be placed inside this function.
async (app, args) => {
  // Get the current URL and DOM of the page.
  const url = await app.getURL();
  const dom = await app.getDOM();
  const html = dom.body.innerHTML;

  // Parse the page content using the Mercury parser API.
  const response = await app.mercury.parse(url, html);

  // Custom style for the simplified content.
  // This uses a light style - you can modify it to suit your needs.
  const style = `
  <style>
    .platypush__simplified-container {
      background: #f0f0f0;
    }

    .platypush__simplified-body {
      min-width: 40em;
      max-width: 50em;
      margin: 0 auto;
      padding: 1em;
      font-family: -apple-system, "Segoe UI", Roboto, Oxygen, Ubuntu, Cantarell, "Fira Sans", "Open Sans", "Droid Sans", "Helvetica Neue", Helvetica, Arial, sans-serif;
      font-size: 20px;
      font-weight: 300;
      color: #333;
      background: white;
      line-height: 1.25em;
      text-align: justify;
    }

    .platypush__simplified-body h1 {
      border-bottom: 1px solid black;
      margin-bottom: 1.5em;
      padding-top: 1em;
      padding-bottom: 0.5em;
      font-size: 1.5em;
      line-height: 1.25em;
      color: #333;
    }

    .platypush__simplified-body p,
    .platypush__simplified-body ul,
    .platypush__simplified-body li,
    .platypush__simplified-body h2,
    .platypush__simplified-body h3,
    .platypush__simplified-body h4,
    .platypush__simplified-body h5 {
      color: #333 !important;
    }

    .platypush__simplified-body a {
      color: #4078c0 !important;
    }

    .platypush__simplified-body p {
      margin-bottom: 1.5em;
    }

    .platypush__original-body {
      display: none;
    }

    .platypush__simplified-banner {
      position: fixed;
      top: 0.5em;
      left: 0.5em;
      color: black;
      font-size: 2em;
      z-index: 99999;
    }


    @media print {
      .platypush__simplified-banner {
        display: none;
      }
    }
  </style>`;

  // Create a new simplified div with the parsed content.
  // .platypush__simplified-body contains the main content,
  // .platypush__original-body contains the original (hidden) page content.
  // An anchor link is provided to return to the original page content.
  const simplifiedDiv = `
    <div class="platypush__simplified-container">
      <div class="platypush__simplified-banner">
        <a href="#"
          onclick="document.body.innerHTML = document.querySelector('.platypush__original-body').innerHTML; return false;">
          ‹
        </a>
      </div>
      <div class="platypush__simplified-body">
        <h1>${response.title}</h1>
        ${response.content}
      </div>
      <div class="platypush__original-body">
        ${dom.body.innerHTML}
      </div>
    </div>`;

  // Add the custom style and the simplified content to the DOM.
  dom.head.innerHTML += style;
  dom.body.innerHTML = simplifiedDiv;

  // Set the modified DOM back to the app.
  await app.setDOM(`<html>${dom.getElementsByTagName('html')[0].innerHTML}</html>`);
}