#!/bin/bash

# This script generates a Dockerfile and a docker-compose.yml file for running
# Mopidy with various extensions.
#
# This is customized for my setup, so you may need to adjust it to fit the
# extensions that you actually need.
#
# You may also want to use a Pulse sink for audio output if you already have
# PulseAudio running on the host system, otherwise ALSA won't to
# control the audio output.
#
# Also note that the docker-compose.yml will mount the configuration and work
# directories from the host system ($HOME/.config/mopidy and
# $HOME/.local/share/mopidy).

cat <<EOF > Dockerfile
FROM debian:12-slim
LABEL Author="Fabio Manganiello"

# Change the architecture if needed, e.g. arm64, armhf, etc.
ARG ARCH=amd64

ENV DEBIAN_FRONTEND=noninteractive
ENV PIP_NO_CACHE_DIR=1

RUN apt update && apt upgrade -y \\
 && apt install -y \\
    alsa-utils \\
    cmake \\
    curl \\
    dumb-init \\
    gstreamer1.0-alsa \\
    gstreamer1.0-pulseaudio \\
    libcairo2-dev \\
    mopidy \\
    pkg-config \\
    pulseaudio-utils \\
    python3-pip \\
    sudo \\
 && mkdir -p ~/.config/pip && printf "[global]\nbreak-system-packages = true\n" | tee ~/.config/pip/pip.conf \\
 && pip install --no-cache-dir \\
    tidalapi==0.8.3 \\
    mopidy-bandcamp \\
    mopidy-funkwhale \\
    mopidy-iris \\
    mopidy-jellyfin \\
    mopidy-local \\
    mopidy-mobile \\
    mopidy-mpd \\
    mopidy-podcast \\
    mopidy-scrobbler \\
    mopidy-soundcloud \\
    mopidy-tidal \\
    mopidy-tunein \\
    mopidy-youtube \\
    yt-dlp \\
 && apt purge --auto-remove -y \\
    cmake \\
    gcc \\
    libcairo2-dev \\
    pkg-config \\
 && apt clean \\
 && apt autoremove -y \\
 && rm -rf /var/lib/apt/lists/* /tmp/* /var/tmp/* ~/.cache /root/.cache

# Needed to map the audio group to the host audio group,
# and that the mopidy user is part of it, and that the user has the same UID as the host user.
RUN usermod -u $(id -u) mopidy
RUN groupmod -g $(grep audio /etc/group | sed -r -e 's/^([^:]+:){2}([0-9]+):.*/\2/') audio
RUN usermod -aG audio mopidy
RUN sh -c 'echo "mopidy ALL=NOPASSWD: /usr/local/lib/python3.11/dist-packages/mopidy_iris/system.sh" >> /etc/sudoers'

USER mopidy

EXPOSE 6600
EXPOSE 6680
EXPOSE 5555/udp

ADD ./start.sh /start.sh

ENTRYPOINT ["dumb-init", "--"]
CMD ["/start.sh"]

HEALTHCHECK --interval=60s --timeout=5s --retries=3 \\
    CMD curl --connect-timeout 5 --silent --show-error --fail http://localhost:6680/ || exit 1
EOF

cat <<EOF > docker-compose.yml
services:
  mopidy:
    build:
      context: .
      dockerfile: Dockerfile
    container_name: mopidy
    ports:
      - "6600:6600"  # MPD port
      - "6680:6680"  # HTTP port
      - "5555:5555"  # Streaming UDP port
    devices:
      - /dev/snd:/dev/snd  # Audio device
    hostname: mopidy
    environment:
      PULSE_SERVER: unix:/run/user/$(id -u)/pulse/native  # Adjust user ID as needed
    volumes:
      - /run/user/$(id -u)/pulse:/run/user/$(id -u)/pulse  # Adjust user ID as needed
      - /run/user/$(id -u)/pulse/native:/run/user/$(id -u)/pulse/native  # Adjust user ID as needed
      # - /mnt/hd/media/music:/mnt/hd/media/music  # Music library path
      - $HOME/.config/mopidy/mopidy.conf:/mopidy/mopidy.conf  # Configuration file
      - $HOME/.local/share/mopidy:/var/lib/mopidy/.local/share/mopidy  # Local data storage
      - ./cache:/var/lib/mopidy/.cache/mopidy
    restart: unless-stopped
EOF
